<?php
/*--------------------------------------------------------------
   GambioWithdrawalsCount.php 2021-08-19
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition;

use Doctrine\DBAL\Connection;
use Gambio\Admin\Modules\Statistics\App\Data\Factory as DataFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory as OverviewFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Option\Predefined\TimespanDropdown;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Options;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data;
use NumberFormatter;

class GambioWithdrawalsCount extends GambioWidgetDefinition
{
    private const TYPE = 'gambio.withdrawals.count';
    
    private const VERSION = '0.1.0';
    
    private const ICON = "data:image/png;base64,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";
    
    private const WIDGET_TITLE = [
        self::LANGUAGE_CODE_GERMAN  => 'Widerrufe',
        self::LANGUAGE_CODE_ENGLISH => 'Withdrawals',
    ];
    
    /**
     * @var DataFactory
     */
    private $dataFactory;
    
    /**
     * @var Connection
     */
    private $connection;
    
    /**
     * @var NumberFormatter
     */
    private $numberFormatter;
    
    
    public function __construct(
        OverviewFactory $overviewFactory,
        DataFactory     $dataFactory,
        Connection      $connection,
        NumberFormatter $numberFormatter
    ) {
        $this->dataFactory     = $dataFactory;
        $this->connection      = $connection;
        $this->numberFormatter = $numberFormatter;
        
        parent::__construct($overviewFactory->createType(self::TYPE),
                            $overviewFactory->createVersion(self::VERSION),
                            $overviewFactory->createIconUsingData(self::ICON),
                            $overviewFactory->useVisualizations()->createText(),
                            $overviewFactory->useOptions()->createOptions($overviewFactory->useOptions()
                                                                              ->usePredefined()
                                                                              ->createTimespanDropdownIncludingToday($overviewFactory)),
                            $overviewFactory->createTitles($overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_GERMAN),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_GERMAN]),
                                                           $overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_ENGLISH),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_ENGLISH])));
    }
    
    
    /**
     * @inheritDoc
     */
    public function data(Options $options): Data
    {
        $timespan    = $this->dataFactory->useTimespan()->createFromTerm($options->getById(TimespanDropdown::ID)
                                                                             ->value());
        $withdrawals = $this->connection->createQueryBuilder()
                           ->select('COUNT(*) AS withdrawals')
                           ->from('withdrawals')
                           ->where('withdrawal_date BETWEEN :startDate AND :endDate')
                           ->setParameters([
                                               ':startDate' => $timespan->startDate()
                                                   ->format(self::QUERY_TIMESPAN_FORMAT_DATE_START),
                                               ':endDate'   => $timespan->endDate()
                                                   ->format(self::QUERY_TIMESPAN_FORMAT_DATE_END),
                                           ])
                           ->execute()
                           ->fetchAll()[0];
        
        return $this->dataFactory->useTextData()->createTextData($this->dataFactory->useTextData()
                                                                     ->createValue($this->numberFormatter->format((int)($withdrawals['withdrawals']
                                                                                                                        ??
                                                                                                                        0))));
    }
}